# LIME2025テーマ - 100点を目指す改善方針

## 現状分析と課題

現在のLIME2025テーマは、LINE風デザイン言語を取り入れようとしていますが、以下の課題があります：

- **デザインの一貫性不足**: LINE風とRedmineのデフォルトスタイルが混在
- **コンポーネントの不整合**: 同じ機能のUI要素でもスタイルがバラバラ
- **トンマナの混在**: LINE独特のデザイン言語が部分的にしか適用されていない
- **レイアウトの崩れ**: CSSの優先度や継承の問題で一部スタイルが上書きされない

現状の評価：**40点/100点**

## 改善方針と重点ポイント

### 1. デザイン言語の統一 (30点アップ)

LINEアプリの特徴的なデザイン言語を理解し、Redmine全体に一貫して適用します。

#### 具体的な実装案

```css
/* 統一されたLINE風変数定義 */
:root {
  /* コアカラー */
  --line-green: #06C755;
  --line-green-dark: #05b14c;
  --line-green-light: rgba(6, 199, 85, 0.1);
  
  /* 背景色と表面色 */
  --line-background: #F7F8FA;  /* LINEアプリの特徴的な薄いグレー背景 */
  --line-card: #FFFFFF;
  
  /* テキストカラー */
  --line-text-primary: #1E1E1E;
  --line-text-secondary: #8A8F98;
  
  /* 装飾要素 */
  --line-shadow: 0 2px 4px rgba(0, 0, 0, 0.08);
  --line-radius: 8px;
  --line-border: #EBEEF4;
  
  /* アクセントカラー */
  --line-error: #FF334B;
  --line-warning: #FF8F00;
  --line-info: #35A0EE;
}

/* ベーススタイル */
body {
  background-color: var(--line-background);
  color: var(--line-text-primary);
  font-family: -apple-system, BlinkMacSystemFont, 'Hiragino Sans', 'Hiragino Kaku Gothic ProN', 'Noto Sans JP', sans-serif;
  line-height: 1.5;
  margin: 0;
  padding: 0;
}
```

#### チェックポイント
- [ ] すべてのカラー値をLINE風に統一
- [ ] コンポーネントごとに適切な変数を使用
- [ ] 余白、シャドウ、境界線の一貫性を確保

### 2. フォントとタイポグラフィの統一 (15点アップ)

LINEアプリで使用されているフォントとタイポグラフィのスタイルを正確に模倣します。

#### 具体的な実装案

```css
/* フォントサイズの一貫したスケール */
:root {
  --line-font-xs: 12px;
  --line-font-sm: 14px;
  --line-font-md: 16px; 
  --line-font-lg: 18px;
  --line-font-xl: 20px;
  --line-font-xxl: 24px;
  
  /* LINE特有の太さ */
  --line-weight-normal: 400;
  --line-weight-medium: 500;
  --line-weight-bold: 700;
}

/* 基本フォント設定 */
body, input, button, select, textarea {
  font-family: -apple-system, BlinkMacSystemFont, 'Hiragino Sans', 'Hiragino Kaku Gothic ProN', 'Noto Sans JP', sans-serif;
  font-size: var(--line-font-md);
  font-weight: var(--line-weight-normal);
}

/* 見出し */
h1 {
  font-size: var(--line-font-xxl);
  font-weight: var(--line-weight-bold);
  margin-bottom: 16px;
}

h2 {
  font-size: var(--line-font-xl);
  font-weight: var(--line-weight-bold);
  margin-bottom: 12px;
}

h3 {
  font-size: var(--line-font-lg);
  font-weight: var(--line-weight-bold);
  margin-bottom: 8px;
}

/* メニュー項目 */
.menu-item, .button {
  font-weight: var(--line-weight-medium);
}
```

#### チェックポイント
- [ ] すべてのテキスト要素で統一したフォントファミリーを使用
- [ ] LINE独自のフォントウェイト分布を再現
- [ ] 階層的なフォントサイズを一貫して適用

### 3. ナビゲーション・メニューの完全模倣 (15点アップ)

LINEアプリのボトムナビゲーションバーのデザインを正確に再現します。

#### 具体的な実装案

```css
/* LINE風ボトムナビゲーション */
#main-menu {
  display: flex;
  height: 56px;
  background-color: var(--line-card);
  border-top: 1px solid var(--line-border);
  padding: 0;
  position: fixed;
  bottom: 0;
  left: 0;
  right: 0;
  z-index: 1000;
}

#main-menu ul {
  display: flex;
  width: 100%;
  margin: 0;
  padding: 0;
  list-style: none;
}

#main-menu li {
  flex: 1;
  text-align: center;
  margin: 0;
}

#main-menu li a {
  height: 100%;
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  color: var(--line-text-secondary);
  transition: color 0.2s;
  border-bottom: none;
  padding: 8px 0;
  text-decoration: none;
}

#main-menu li a:hover {
  background-color: transparent;
  color: var(--line-green);
}

#main-menu li a.selected {
  color: var(--line-green);
  background-color: transparent;
  border-bottom: none;
}

/* アイコンを上部に、テキストを下部に配置 */
#main-menu li a .menu-icon {
  display: block;
  margin: 0 auto 4px;
  width: 24px;
  height: 24px;
}

/* メニューラベル */
#main-menu li a .menu-label {
  font-size: var(--line-font-xs);
  font-weight: var(--line-weight-medium);
}
```

#### JavaScriptでのアイコンと構造変更

```javascript
function transformMainMenu() {
  const mainMenu = document.querySelector('#main-menu');
  if (!mainMenu) return;
  
  // メニュー項目を取得
  const menuItems = mainMenu.querySelectorAll('li a');
  
  menuItems.forEach(item => {
    // 現在のテキストを保存
    const text = item.textContent.trim();
    
    // メニュー項目の内容をクリア
    item.innerHTML = '';
    
    // アイコン用の要素を追加
    const iconSpan = document.createElement('span');
    iconSpan.className = 'menu-icon';
    
    // テキスト用の要素を追加
    const labelSpan = document.createElement('span');
    labelSpan.className = 'menu-label';
    labelSpan.textContent = text;
    
    // アイコンと名前を追加
    item.appendChild(iconSpan);
    item.appendChild(labelSpan);
    
    // アイコンを選択
    const iconName = determineIconFromText(text);
    iconSpan.innerHTML = getIconSVG(iconName);
  });
}
```

#### チェックポイント
- [ ] メニューがLINEアプリと同じレイアウトで表示される
- [ ] アイコンが上、テキストが下の配置になっている
- [ ] 選択状態のスタイルがLINE風になっている

### 4. カードとコンテナのスタイル統一 (15点アップ)

LINEアプリのカードデザインを全てのコンテナ要素に統一して適用します。

#### 具体的な実装案

```css
/* LINE風カードコンポーネント */
.card-component {
  background-color: var(--line-card);
  border-radius: var(--line-radius);
  box-shadow: var(--line-shadow);
  border: none;
  margin-bottom: 16px;
  overflow: hidden;
}

/* カードヘッダー */
.card-header {
  padding: 16px;
  border-bottom: 1px solid var(--line-border);
  font-weight: var(--line-weight-bold);
}

/* カードコンテンツ */
.card-content {
  padding: 16px;
}

/* カードフッター */
.card-footer {
  padding: 12px 16px;
  border-top: 1px solid var(--line-border);
  background-color: rgba(0, 0, 0, 0.02);
}

/* Redmineの主要コンテナにカードスタイルを適用 */
#content, .box, .wiki, .issue, #sidebar h3, .flash {
  background-color: var(--line-card);
  border-radius: var(--line-radius);
  box-shadow: var(--line-shadow);
  border: none;
  overflow: hidden;
}

/* コンテンツエリア特有のスタイル */
#content {
  padding: 0;
}

#content > h2 {
  padding: 16px;
  margin: 0;
  border-bottom: 1px solid var(--line-border);
}

#content > .content-body {
  padding: 16px;
}
```

#### チェックポイント
- [ ] すべてのコンテナがLINE風のカードデザインになっている
- [ ] 角丸、影、余白が一貫している
- [ ] ヘッダー、コンテンツ、フッターの区切りがLINE風になっている

### 5. ボタンとフォーム要素の完全改良 (15点アップ)

LINEアプリのシンプルで統一されたボタンとフォームコントロールを実装します。

#### 具体的な実装案

```css
/* LINE風のプライマリボタン */
input[type="submit"], button, .button {
  background-color: var(--line-green);
  color: white;
  border-radius: 100px; /* LINE特有の丸いボタン */
  height: 40px;
  padding: 0 24px;
  font-weight: var(--line-weight-bold);
  font-size: var(--line-font-md);
  border: none;
  cursor: pointer;
  transition: background-color 0.2s;
  box-shadow: none;
  text-transform: none;
  display: inline-flex;
  align-items: center;
  justify-content: center;
}

input[type="submit"]:hover, button:hover, .button:hover {
  background-color: var(--line-green-dark);
}

/* セカンダリボタン */
.button-secondary {
  background-color: white;
  color: var(--line-green);
  border: 1px solid var(--line-green);
}

.button-secondary:hover {
  background-color: var(--line-green-light);
  color: var(--line-green);
}

/* フォーム要素 */
input[type="text"], 
input[type="password"], 
input[type="search"],
input[type="email"],
input[type="number"],
select, 
textarea {
  border: 1px solid var(--line-border);
  border-radius: var(--line-radius);
  padding: 12px;
  font-size: var(--line-font-md);
  width: 100%;
  max-width: 100%;
  box-sizing: border-box;
  transition: border-color 0.2s;
}

input[type="text"]:focus,
input[type="password"]:focus,
input[type="search"]:focus,
input[type="email"]:focus,
input[type="number"]:focus,
select:focus,
textarea:focus {
  border-color: var(--line-green);
  box-shadow: none;
  outline: none;
}

/* チェックボックスとラジオボタン */
input[type="checkbox"],
input[type="radio"] {
  width: 20px;
  height: 20px;
  margin-right: 8px;
  accent-color: var(--line-green);
}

/* ラベル */
label {
  display: block;
  margin-bottom: 8px;
  font-weight: var(--line-weight-medium);
  color: var(--line-text-primary);
}
```

#### チェックポイント
- [ ] ボタンが LINE アプリの丸みを帯びたデザインになっている
- [ ] フォーム要素のスタイルが統一されている
- [ ] フォーカス状態のスタイルがLINE風になっている
- [ ] チェックボックスとラジオボタンがLINE風のカラーになっている

### 6. アイコン統一とLINE風アイコンセットの導入 (10点アップ)

LINE風の単色でシンプルなアイコンセットを一貫して使用します。

#### 具体的な実装案

```css
/* LINE風アイコンの基本スタイル */
.line-icon {
  display: inline-block;
  width: 24px;
  height: 24px;
  stroke: currentColor;
  stroke-width: 2;
  stroke-linecap: round;
  stroke-linejoin: round;
  fill: none;
  vertical-align: middle;
}

/* 選択された状態のアイコン */
.selected .line-icon {
  fill: var(--line-green);
}

/* アイコンサイズバリエーション */
.line-icon.small {
  width: 16px;
  height: 16px;
}

.line-icon.large {
  width: 32px;
  height: 32px;
}
```

#### JavaScriptでのアイコン置換

```javascript
// LINE風アイコンセット
const LINE_ICONS = {
  home: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" class="line-icon"><path d="M10 20v-6h4v6h5v-8h3L12 3 2 12h3v8z"/></svg>',
  chat: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" class="line-icon"><path d="M20 2H4c-1.1 0-1.99.9-1.99 2L2 22l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm-2 12H6v-2h12v2zm0-3H6V9h12v2zm0-3H6V6h12v2z"/></svg>',
  // 他のアイコンも同様に定義
};

function replaceIconsWithLineSVG() {
  // アイコンクラスを持つ要素を探す
  document.querySelectorAll('.icon').forEach(icon => {
    // クラス名からアイコンタイプを判断
    for (const [name, svg] of Object.entries(LINE_ICONS)) {
      if (icon.classList.contains(`icon-${name}`)) {
        icon.innerHTML = svg;
        break;
      }
    }
  });
}
```

#### チェックポイント
- [ ] すべてのアイコンがLINE風の統一スタイルになっている
- [ ] アイコンの色や大きさが一貫している
- [ ] 選択状態のアイコンデザインがLINE風になっている

## 実装戦略

### フェーズ1: 設計と基礎の構築

1. LINEアプリのデザイン分析とスクリーンショット収集
2. デザイン変数とベーススタイルの定義
3. 基本レイアウト構造の改良

### フェーズ2: コアコンポーネントの実装

1. ナビゲーションとメニューの再構築
2. カードとコンテナのデザイン適用
3. ボタンとフォーム要素の改良

### フェーズ3: 細部の改善と最適化

1. アイコンの統一と置き換え
2. タイポグラフィの微調整
3. レスポンシブデザインの強化
4. ブラウザ互換性のテスト

## LINE風デザインの主要特徴まとめ

### 視覚的特徴
- **カラー**: 鮮やかな緑（#06C755）をアクセントカラーとして使用
- **背景**: 薄いグレー（#F7F8FA）の背景と白のカード
- **余白**: ゆったりとした余白設計
- **角丸**: すべての要素に一貫した角丸（8px）を適用
- **影**: 軽い影効果で要素に浮遊感を付与

### レイアウト特徴
- **カード方式**: 情報をカードとして表示
- **垂直リスト**: 情報を縦に整理したリスト表示
- **ボトムナビ**: 主要メニューを画面下部に配置
- **アイコン**: シンプルな単色アイコンを一貫して使用

### インタラクション特徴
- **タップフィードバック**: 軽いリップル効果や色の変化
- **アニメーション**: 控えめで自然なトランジション効果
- **フォーカス状態**: 明確だが控えめなフォーカス表示

## ユーザーテスト項目

実装後に以下の項目をチェックして、LINE風デザインの品質を評価します：

1. 一貫性: すべての要素がLINE風デザイン言語に従っているか
2. 使いやすさ: 操作が直感的で簡単か
3. 視認性: 情報が明確に区別され、読みやすいか
4. レスポンシブ性: 様々な画面サイズで正しく表示されるか
5. パフォーマンス: 重いCSSやJSで動作が遅くなっていないか

これらの改善を実施することで、現在の40点から100点レベルの完璧なLINE風デザインを実現できます。
