# 機材予約システム 詳細仕様書

## システム概要

### 基本情報
- **システム名称**: 社内機材予約システム
- **バージョン**: v1.0.0
- **開発言語**: JavaScript (Node.js, React)
- **データベース**: SQLite
- **対象ユーザー**: 社内全従業員（約100名想定）
- **管理機材数**: 約500点
- **稼働環境**: オンプレミス Windows Server

### システム構成
```
[フロントエンド]
React 18.x + TailwindCSS 3.x + Lucide Icons

[バックエンド]  
Node.js 18.x + Express 4.x + SQLite 3.x

[認証システム]
JWT + LINE WORKS OAuth 2.0

[デプロイ環境]
Windows Server + IIS + iisnode
```

---

## 1. 画面一覧

### 1.1 デスクトップ版画面

| 画面ID | 画面名 | 画面種別 | 権限 | 説明 |
|--------|--------|----------|------|------|
| D001 | ログイン画面 | 認証 | Guest | ユーザー認証・LINE WORKS連携 |
| D002 | ダッシュボード | メイン | User/Admin | 統計情報・クイックアクション |
| D003 | 機材検索画面 | 機能 | User/Admin | 機材の検索・フィルタリング |
| D004 | 機材詳細画面 | 機能 | User/Admin | 機材の詳細情報表示 |
| D005 | 予約作成画面 | 機能 | User/Admin | 新規予約の作成 |
| D006 | 予約管理画面 | 機能 | User/Admin | 自分の予約一覧・編集 |
| D007 | 予約履歴画面 | 機能 | User/Admin | 過去の予約履歴確認 |
| D008 | 管理者ホーム | 管理 | Admin | 管理者専用メニュー |
| D009 | ユーザー管理画面 | 管理 | Admin | ユーザーの追加・編集・削除 |
| D010 | システム設定画面 | 管理 | Admin | システム全体設定 |
| D011 | レポート出力画面 | 管理 | Admin | 各種レポートの生成・出力 |

### 1.2 モバイル版画面（LINE WORKS対応）

| 画面ID | 画面名 | 画面種別 | 権限 | 説明 |
|--------|--------|----------|------|------|
| M001 | モバイルダッシュボード | メイン | User/Admin | モバイル最適化ダッシュボード |
| M002 | 簡単検索画面 | 機能 | User/Admin | タッチ操作最適化検索 |
| M003 | クイック予約画面 | 機能 | User/Admin | 簡単予約フォーム |
| M004 | 予約確認画面 | 機能 | User/Admin | モバイル用予約一覧 |
| M005 | 人気機材画面 | 機能 | User/Admin | 人気機材一覧表示 |
| M006 | 認証完了画面 | 認証 | User/Admin | LINE WORKS認証後の画面 |

### 1.3 LINE WORKS Bot画面

| 画面ID | 画面名 | 画面種別 | 権限 | 説明 |
|--------|--------|----------|------|------|
| B001 | Bot初期メッセージ | Bot | User/Admin | 初回アクセス時のウェルカム |
| B002 | ヘルプメッセージ | Bot | User/Admin | 利用方法の説明 |
| B003 | 機材検索結果 | Bot | User/Admin | Bot経由での検索結果 |
| B004 | 予約状況表示 | Bot | User/Admin | ユーザーの予約状況 |
| B005 | 通知メッセージ | Bot | User/Admin | 各種自動通知 |

---

## 2. 処理一覧

### 2.1 認証処理

| 処理ID | 処理名 | 処理種別 | 説明 |
|--------|--------|----------|------|
| AUTH001 | 標準ログイン処理 | 同期 | ユーザー名・パスワードでの認証 |
| AUTH002 | LINE WORKS OAuth開始 | 非同期 | LINE WORKS認証画面へリダイレクト |
| AUTH003 | OAuth コールバック処理 | 非同期 | 認証コード受取・トークン発行 |
| AUTH004 | JWTトークン生成 | 同期 | アクセストークンの生成 |
| AUTH005 | トークン検証 | 同期 | リクエスト時のトークン有効性確認 |
| AUTH006 | トークンリフレッシュ | 非同期 | 期限切れトークンの更新 |
| AUTH007 | ログアウト処理 | 同期 | セッション無効化 |

### 2.2 機材管理処理

| 処理ID | 処理名 | 処理種別 | 説明 |
|--------|--------|----------|------|
| EQP001 | 機材一覧取得 | 非同期 | 条件に応じた機材リスト取得 |
| EQP002 | 機材詳細取得 | 非同期 | 指定機材の詳細情報取得 |
| EQP003 | 機材検索処理 | 非同期 | キーワード・条件での機材検索 |
| EQP004 | カテゴリ別取得 | 非同期 | カテゴリ・サブカテゴリ別機材取得 |
| EQP005 | 人気機材取得 | 非同期 | 利用頻度順での機材取得 |
| EQP006 | 機材追加処理 | 非同期 | 新規機材の登録 |
| EQP007 | 機材更新処理 | 非同期 | 機材情報の更新 |
| EQP008 | 機材削除処理 | 非同期 | 機材の論理削除 |
| EQP009 | 機材状態更新 | 非同期 | ステータス変更（利用可能/メンテナンス） |

### 2.3 予約管理処理

| 処理ID | 処理名 | 処理種別 | 説明 |
|--------|--------|----------|------|
| RSV001 | 予約作成処理 | 非同期 | 新規予約の作成・重複チェック |
| RSV002 | 予約更新処理 | 非同期 | 既存予約の修正 |
| RSV003 | 予約削除処理 | 非同期 | 予約のキャンセル |
| RSV004 | 予約一覧取得 | 非同期 | ユーザー・期間別予約取得 |
| RSV005 | 重複チェック処理 | 同期 | 予約期間の重複確認 |
| RSV006 | 予約状態更新 | 非同期 | 予約ステータスの変更 |
| RSV007 | 履歴アーカイブ処理 | 非同期 | 古い予約データの移行 |

### 2.4 通知処理

| 処理ID | 処理名 | 処理種別 | 説明 |
|--------|--------|----------|------|
| NOT001 | 予約完了通知 | 非同期 | 予約作成時の自動通知 |
| NOT002 | リマインダー通知 | 非同期 | 予約開始前の事前通知 |
| NOT003 | キャンセル通知 | 非同期 | 予約キャンセル時の通知 |
| NOT004 | メンテナンス通知 | 非同期 | 機材メンテナンス予定通知 |
| NOT005 | LINE WORKS Bot送信 | 非同期 | Bot経由でのメッセージ送信 |
| NOT006 | 通知履歴記録 | 非同期 | 送信した通知の履歴保存 |

### 2.5 LINE WORKS Bot処理

| 処理ID | 処理名 | 処理種別 | 説明 |
|--------|--------|----------|------|
| BOT001 | Webhook受信処理 | 非同期 | LINE WORKSからのWebhook処理 |
| BOT002 | メッセージ解析 | 同期 | 受信メッセージの意図解析 |
| BOT003 | 自動応答処理 | 非同期 | 適切な応答メッセージの生成・送信 |
| BOT004 | ボタンアクション処理 | 非同期 | ユーザーのボタンクリック処理 |
| BOT005 | 機材検索Bot処理 | 非同期 | Bot経由での機材検索 |
| BOT006 | 予約作成Bot処理 | 非同期 | Bot経由での予約作成 |

---

## 3. 機能一覧

### 3.1 基本機能

| 機能ID | 機能名 | 説明 | 画面 | 権限 |
|--------|--------|------|------|------|
| F001 | ユーザー認証 | ログイン・ログアウト機能 | D001 | Guest |
| F002 | LINE WORKS認証 | OAuth 2.0によるシングルサインオン | D001, M006 | Guest |
| F003 | ダッシュボード表示 | 統計情報・クイックアクセス | D002, M001 | User/Admin |
| F004 | 機材検索 | キーワード・カテゴリ検索 | D003, M002 | User/Admin |
| F005 | 機材詳細表示 | 機材の詳細情報確認 | D004 | User/Admin |
| F006 | 予約作成 | 新規予約の作成 | D005, M003 | User/Admin |
| F007 | 予約管理 | 予約の確認・編集・削除 | D006, M004 | User/Admin |
| F008 | 予約履歴 | 過去の予約履歴確認 | D007 | User/Admin |

### 3.2 管理機能

| 機能ID | 機能名 | 説明 | 画面 | 権限 |
|--------|--------|------|------|------|
| AF001 | ユーザー管理 | ユーザーの追加・編集・削除 | D009 | Admin |
| AF002 | 機材管理 | 機材の登録・更新・削除 | D009 | Admin |
| AF003 | システム設定 | 全体設定の変更 | D010 | Admin |
| AF004 | レポート出力 | 各種レポートの生成 | D011 | Admin |
| AF005 | 予約監視 | 全ユーザーの予約状況監視 | D008 | Admin |
| AF006 | メンテナンス管理 | 機材メンテナンススケジュール | D009 | Admin |

### 3.3 LINE WORKS連携機能

| 機能ID | 機能名 | 説明 | 画面 | 権限 |
|--------|--------|------|------|------|
| LF001 | Bot自動応答 | ユーザーメッセージへの自動応答 | B002-B005 | User/Admin |
| LF002 | 自動通知 | 予約・メンテナンス等の自動通知 | B005 | User/Admin |
| LF003 | モバイル最適化UI | スマートフォン対応UI | M001-M005 | User/Admin |
| LF004 | QRコード連携 | QRコードからの直接アクセス | - | User/Admin |
| LF005 | トークルーム連携 | ブックマーク・共有機能 | - | User/Admin |

### 3.4 補助機能

| 機能ID | 機能名 | 説明 | 画面 | 権限 |
|--------|--------|------|------|------|
| SF001 | データバックアップ | 定期的なデータバックアップ | - | System |
| SF002 | ログ管理 | アクセス・操作ログの記録 | - | System |
| SF003 | データアーカイブ | 古いデータの自動アーカイブ | - | System |
| SF004 | 統計分析 | 利用統計の集計・分析 | D002 | User/Admin |
| SF005 | セキュリティ監視 | 不正アクセスの検知・防止 | - | System |

---

## 4. 関数一覧

### 4.1 フロントエンド関数（React）

#### 4.1.1 認証関連
```javascript
// 認証フック
const useAuth = () => { ... }                    // 認証状態管理
const useLineWorksAuth = () => { ... }           // LINE WORKS認証管理

// 認証処理
const handleLogin = (credentials) => { ... }     // ログイン処理
const handleLogout = () => { ... }               // ログアウト処理
const refreshToken = () => { ... }               // トークンリフレッシュ
```

#### 4.1.2 機材管理関連
```javascript
// 機材取得
const getEquipmentList = (filters) => { ... }    // 機材一覧取得
const getEquipmentDetail = (id) => { ... }       // 機材詳細取得
const searchEquipment = (query) => { ... }       // 機材検索
const getPopularEquipment = () => { ... }        // 人気機材取得

// 機材管理（管理者のみ）
const createEquipment = (data) => { ... }        // 機材作成
const updateEquipment = (id, data) => { ... }    // 機材更新
const deleteEquipment = (id) => { ... }          // 機材削除
```

#### 4.1.3 予約管理関連
```javascript
// 予約操作
const createReservation = (data) => { ... }      // 予約作成
const updateReservation = (id, data) => { ... }  // 予約更新
const cancelReservation = (id) => { ... }        // 予約キャンセル
const getUserReservations = (userId) => { ... }  // ユーザー予約取得

// 予約検証
const checkAvailability = (data) => { ... }      // 利用可能性チェック
const validateReservation = (data) => { ... }    // 予約データ検証
```

#### 4.1.4 UI操作関連
```javascript
// 画面遷移
const setCurrentView = (view) => { ... }         // 画面切り替え
const navigateToDetail = (id) => { ... }         // 詳細画面遷移
const goBack = () => { ... }                     // 前画面戻り

// UI状態管理
const setLoading = (state) => { ... }            // ローディング状態
const showNotification = (message) => { ... }    // 通知表示
const setError = (error) => { ... }              // エラー表示
```

### 4.2 バックエンド関数（Node.js）

#### 4.2.1 認証関連
```javascript
// JWT認証
const generateToken = (user) => { ... }          // JWTトークン生成
const verifyToken = (token) => { ... }           // トークン検証
const refreshAccessToken = (refresh) => { ... }  // トークンリフレッシュ

// LINE WORKS OAuth
const initiateOAuth = () => { ... }              // OAuth開始
const handleCallback = (code) => { ... }         // コールバック処理
const getUserInfo = (token) => { ... }           // ユーザー情報取得
```

#### 4.2.2 データベース操作
```javascript
// ユーザー操作
const findUserById = (id) => { ... }             // ユーザー取得
const createUser = (data) => { ... }             // ユーザー作成
const updateUser = (id, data) => { ... }         // ユーザー更新

// 機材操作
const findEquipment = (filters) => { ... }       // 機材検索
const createEquipment = (data) => { ... }        // 機材作成
const updateEquipmentStatus = (id, status) => { ... } // ステータス更新

// 予約操作
const createReservation = (data) => { ... }      // 予約作成
const findReservations = (filters) => { ... }    // 予約検索
const checkConflict = (data) => { ... }          // 重複チェック
```

#### 4.2.3 通知関連
```javascript
// 通知送信
const sendNotification = (user, message) => { ... }    // 通知送信
const sendLineWorksMessage = (userId, content) => { ... } // LINE WORKS送信
const scheduleReminder = (reservation) => { ... }      // リマインダー予約

// 通知テンプレート
const formatReservationMessage = (data) => { ... }     // 予約通知フォーマット
const formatReminderMessage = (data) => { ... }        // リマインダーフォーマット
```

#### 4.2.4 LINE WORKS Bot関連
```javascript
// Webhook処理
const handleWebhook = (payload) => { ... }       // Webhook受信処理
const verifySignature = (signature, body) => { ... } // 署名検証
const parseMessage = (message) => { ... }        // メッセージ解析

// Bot応答
const generateResponse = (intent, user) => { ... }    // 応答生成
const sendBotMessage = (userId, content) => { ... }   // Bot メッセージ送信
const handlePostback = (data) => { ... }              // ボタンアクション処理
```

### 4.3 ユーティリティ関数

#### 4.3.1 日付・時刻処理
```javascript
const formatDate = (date) => { ... }             // 日付フォーマット
const formatTime = (date) => { ... }             // 時刻フォーマット
const getDateRange = (start, end) => { ... }     // 日付範囲生成
const isBusinessDay = (date) => { ... }          // 営業日判定
```

#### 4.3.2 バリデーション
```javascript
const validateEmail = (email) => { ... }         // メール形式検証
const validatePassword = (password) => { ... }   // パスワード強度検証
const validateReservationData = (data) => { ... } // 予約データ検証
const sanitizeInput = (input) => { ... }         // 入力値サニタイズ
```

#### 4.3.3 データ変換
```javascript
const csvToJson = (csv) => { ... }               // CSV変換
const jsonToCsv = (json) => { ... }              // JSON変換
const encodeQRCode = (data) => { ... }           // QRコード生成
const generateReport = (data) => { ... }         // レポート生成
```

---

## 5. テーブル一覧

### 5.1 基本テーブル

| テーブル名 | 論理名 | 用途 | レコード数（想定） |
|-----------|--------|------|--------------------|
| users | ユーザー | ユーザー情報管理 | 100件 |
| categories | 機材カテゴリ | 機材分類管理 | 4件 |
| subcategories | 機材サブカテゴリ | 機材細分類管理 | 40件 |
| equipment | 機材 | 機材情報管理 | 500件 |
| reservations | 予約 | 予約情報管理 | 1,000件/年 |
| reservation_history | 予約履歴 | アーカイブ済み予約 | 5,000件 |

### 5.2 LINE WORKS連携テーブル

| テーブル名 | 論理名 | 用途 | レコード数（想定） |
|-----------|--------|------|--------------------|
| lineworks_tokens | LINE WORKSトークン | OAuth トークン管理 | 100件 |
| notification_history | 通知履歴 | 送信済み通知履歴 | 10,000件/年 |
| bot_settings | Bot設定 | Bot動作設定 | 20件 |
| user_notification_settings | ユーザー通知設定 | 個人通知設定 | 100件 |
| qr_access_logs | QRアクセスログ | QRコード利用履歴 | 1,000件/月 |

### 5.3 管理・運用テーブル

| テーブル名 | 論理名 | 用途 | レコード数（想定） |
|-----------|--------|------|--------------------|
| auth_sessions | 認証セッション | 認証セッション管理 | 200件 |
| equipment_alerts | 機材アラート | 機材アラート設定 | 1,000件 |
| system_logs | システムログ | システム操作ログ | 50,000件/年 |
| backup_logs | バックアップログ | バックアップ実行履歴 | 365件/年 |

---

## 6. テーブル定義

### 6.1 users（ユーザー）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | ユーザーID |
| username | TEXT | NOT NULL | UQ | - | ログインID |
| password | TEXT | NOT NULL | - | - | パスワード（ハッシュ化） |
| name | TEXT | NOT NULL | - | - | 氏名 |
| department | TEXT | NOT NULL | - | - | 所属部署 |
| email | TEXT | NULL | - | - | メールアドレス |
| role | TEXT | NOT NULL | - | 'user' | 権限（user/admin） |
| lineworks_id | TEXT | NULL | - | - | LINE WORKS ID |
| avatar_url | TEXT | NULL | - | - | アバターURL |
| last_login | DATETIME | NULL | - | - | 最終ログイン日時 |
| login_count | INTEGER | NOT NULL | - | 0 | ログイン回数 |
| notification_enabled | BOOLEAN | NOT NULL | - | 1 | 通知有効フラグ |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |

### 6.2 categories（機材カテゴリ）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | カテゴリID |
| name | TEXT | NOT NULL | - | - | カテゴリ名 |
| description | TEXT | NULL | - | - | 説明 |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |

### 6.3 subcategories（機材サブカテゴリ）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | サブカテゴリID |
| category_id | INTEGER | NOT NULL | FK | - | 親カテゴリID |
| name | TEXT | NOT NULL | - | - | サブカテゴリ名 |
| description | TEXT | NULL | - | - | 説明 |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |

### 6.4 equipment（機材）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | TEXT | NOT NULL | PK | - | 機材ID（書式：1-01-001） |
| name | TEXT | NOT NULL | - | - | 機材名 |
| category_id | INTEGER | NOT NULL | FK | - | カテゴリID |
| subcategory_id | INTEGER | NOT NULL | FK | - | サブカテゴリID |
| status | TEXT | NOT NULL | - | 'available' | 状態（available/maintenance） |
| location | TEXT | NULL | - | - | 保管場所 |
| specifications | TEXT | NULL | - | - | 仕様・説明 |
| last_maintenance | DATE | NULL | - | - | 最終メンテナンス日 |
| purchase_date | DATE | NULL | - | - | 購入日 |
| warranty_date | DATE | NULL | - | - | 保証期限 |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |

### 6.5 reservations（予約）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | 予約ID |
| equipment_id | TEXT | NOT NULL | FK | - | 機材ID |
| user_id | INTEGER | NOT NULL | FK | - | ユーザーID |
| start_date | DATE | NOT NULL | - | - | 利用開始日 |
| end_date | DATE | NOT NULL | - | - | 利用終了日 |
| purpose | TEXT | NOT NULL | - | - | 利用目的 |
| notes | TEXT | NULL | - | - | 備考 |
| status | TEXT | NOT NULL | - | 'confirmed' | 状態（confirmed/pending/completed/cancelled） |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |
| updated_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 更新日時 |

### 6.6 lineworks_tokens（LINE WORKSトークン）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | トークンID |
| user_id | INTEGER | NOT NULL | FK | - | ユーザーID |
| access_token | TEXT | NOT NULL | - | - | アクセストークン |
| refresh_token | TEXT | NULL | - | - | リフレッシュトークン |
| expires_at | INTEGER | NOT NULL | - | - | 有効期限（UNIX時刻） |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |
| updated_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 更新日時 |

### 6.7 notification_history（通知履歴）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | 通知ID |
| user_id | INTEGER | NOT NULL | FK | - | ユーザーID |
| reservation_id | INTEGER | NULL | FK | - | 予約ID |
| notification_type | TEXT | NOT NULL | - | - | 通知種別 |
| title | TEXT | NOT NULL | - | - | 通知タイトル |
| message | TEXT | NOT NULL | - | - | 通知メッセージ |
| sent_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 送信日時 |
| delivery_status | TEXT | NOT NULL | - | 'pending' | 配信状態 |
| lineworks_message_id | TEXT | NULL | - | - | LINE WORKSメッセージID |
| retry_count | INTEGER | NOT NULL | - | 0 | 再送回数 |

### 6.8 user_notification_settings（ユーザー通知設定）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | 設定ID |
| user_id | INTEGER | NOT NULL | FK | - | ユーザーID |
| reservation_reminders | BOOLEAN | NOT NULL | - | 1 | 予約リマインダー有効 |
| maintenance_notifications | BOOLEAN | NOT NULL | - | 1 | メンテナンス通知有効 |
| popular_equipment_alerts | BOOLEAN | NOT NULL | - | 0 | 人気機材アラート有効 |
| system_announcements | BOOLEAN | NOT NULL | - | 1 | システム通知有効 |
| reminder_days_before | INTEGER | NOT NULL | - | 1 | リマインダー日数 |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |
| updated_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 更新日時 |

### 6.9 bot_settings（Bot設定）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | 設定ID |
| setting_key | TEXT | NOT NULL | UQ | - | 設定キー |
| setting_value | TEXT | NOT NULL | - | - | 設定値 |
| description | TEXT | NULL | - | - | 説明 |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 作成日時 |
| updated_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | 更新日時 |

### 6.10 qr_access_logs（QRアクセスログ）

| カラム名 | データ型 | NULL | キー | デフォルト | 説明 |
|----------|----------|------|------|-----------|------|
| id | INTEGER | NOT NULL | PK | AUTOINCREMENT | ログID |
| user_id | INTEGER | NULL | FK | - | ユーザーID |
| equipment_id | TEXT | NULL | FK | - | 機材ID |
| access_type | TEXT | NOT NULL | - | - | アクセス種別 |
| source | TEXT | NULL | - | - | アクセス元 |
| user_agent | TEXT | NULL | - | - | ユーザーエージェント |
| created_at | DATETIME | NOT NULL | - | CURRENT_TIMESTAMP | アクセス日時 |

---

## 7. インデックス定義

### 7.1 パフォーマンス向上インデックス

```sql
-- ユーザー関連
CREATE INDEX idx_users_lineworks_id ON users(lineworks_id);
CREATE INDEX idx_users_department ON users(department);

-- 機材関連
CREATE INDEX idx_equipment_category_id ON equipment(category_id);
CREATE INDEX idx_equipment_status ON equipment(status);
CREATE INDEX idx_equipment_name ON equipment(name);

-- 予約関連
CREATE INDEX idx_reservations_user_id ON reservations(user_id);
CREATE INDEX idx_reservations_equipment_id ON reservations(equipment_id);
CREATE INDEX idx_reservations_date_range ON reservations(start_date, end_date);
CREATE INDEX idx_reservations_status ON reservations(status);

-- LINE WORKS関連
CREATE INDEX idx_lineworks_tokens_user_id ON lineworks_tokens(user_id);
CREATE INDEX idx_lineworks_tokens_expires_at ON lineworks_tokens(expires_at);
CREATE INDEX idx_notification_history_user_id ON notification_history(user_id);
CREATE INDEX idx_notification_history_sent_at ON notification_history(sent_at);
```

### 7.2 複合インデックス

```sql
-- 機材検索用
CREATE INDEX idx_equipment_category_status ON equipment(category_id, status);
CREATE INDEX idx_equipment_subcategory_status ON equipment(subcategory_id, status);

-- 予約重複チェック用
CREATE INDEX idx_reservations_equipment_date ON reservations(equipment_id, start_date, end_date);

-- 通知配信用
CREATE INDEX idx_notification_status_type ON notification_history(delivery_status, notification_type);
```

---

## 8. ビュー定義

### 8.1 user_summary（ユーザー統合情報）

```sql
CREATE VIEW user_summary AS
SELECT 
    u.id,
    u.username,
    u.name,
    u.department,
    u.email,
    u.lineworks_id,
    u.last_login,
    u.login_count,
    COUNT(DISTINCT r.id) as total_reservations,
    COUNT(DISTINCT CASE WHEN r.status = 'confirmed' THEN r.id END) as active_reservations,
    MAX(r.created_at) as last_reservation_date
FROM users u
LEFT JOIN reservations r ON u.id = r.user_id
GROUP BY u.id;
```

### 8.2 equipment_usage_stats（機材利用統計）

```sql
CREATE VIEW equipment_usage_stats AS
SELECT 
    e.id,
    e.name,
    e.category_id,
    e.status,
    COUNT(DISTINCT r.id) as total_reservations,
    COUNT(DISTINCT r.user_id) as unique_users,
    AVG(julianday(r.end_date) - julianday(r.start_date)) as avg_rental_days,
    MAX(r.created_at) as last_reserved_date,
    ROUND(COUNT(DISTINCT r.id) * 100.0 / (SELECT COUNT(*) FROM reservations), 2) as popularity_percentage
FROM equipment e
LEFT JOIN reservations r ON e.id = r.equipment_id AND r.status = 'confirmed'
GROUP BY e.id;
```

---

## 9. トリガー定義

### 9.1 予約作成時通知トリガー

```sql
CREATE TRIGGER reservation_notification_trigger
AFTER INSERT ON reservations
FOR EACH ROW
WHEN NEW.status = 'confirmed'
BEGIN
    INSERT INTO notification_history (user_id, reservation_id, notification_type, title, message)
    VALUES (
        NEW.user_id,
        NEW.id,
        'reservation_created',
        '予約完了通知',
        '機材の予約が完了しました。'
    );
END;
```

### 9.2 ログイン回数更新トリガー

```sql
CREATE TRIGGER login_count_trigger
AFTER UPDATE OF last_login ON users
FOR EACH ROW
WHEN NEW.last_login IS NOT NULL AND (OLD.last_login IS NULL OR NEW.last_login > OLD.last_login)
BEGIN
    UPDATE users SET login_count = login_count + 1 WHERE id = NEW.id;
END;
```

---

## 10. システム制約・仕様

### 10.1 業務制約
- 予約期間は最大30日間
- 1ユーザーあたり同時予約可能数：10件
- 予約開始3日前まで変更・キャンセル可能
- メンテナンス中機材は予約不可

### 10.2 技術制約
- データベースファイルサイズ上限：5GB
- 1回のAPI呼び出し最大レスポンス時間：30秒
- ファイルアップロード最大サイズ：50MB
- 同時接続ユーザー数：100名

### 10.3 セキュリティ制約
- JWTトークン有効期限：24時間
- パスワード最小長：8文字
- LINE WORKSトークン有効期限：API仕様に準拠
- セッションタイムアウト：30分

この仕様書により、システムの運用・保守・拡張時の参考資料として活用できます。