# 仕様検討レビュー・改善提案書

## 1. 一貫性レビュー結果

### 1.1 ✅ 良好な点

| 項目 | 評価 | 詳細 |
|------|------|------|
| **要件の網羅性** | ◎ | システム要件定義から関数・API仕様まで一貫して網羅 |
| **ID管理** | ◎ | 要件ID（REQ-XXX）、機能ID（FUNC-XXX）、処理ID（PROC-XXX）、関数ID（AUTH-XXX等）、API ID（API-XXX）で体系的管理 |
| **多言語対応** | ◎ | 要件からデータベース設計まで一貫して8言語対応を考慮 |
| **モジュール設計** | ◎ | コア機能と拡張機能の分離が明確、柔軟な組み合わせが可能 |
| **外国人配慮** | ◎ | 在留資格、国籍、LINE Bot等、ターゲットユーザーのニーズを反映 |

### 1.2 ⚠️ 改善が必要な点

| 項目 | 課題 | 影響度 |
|------|------|--------|
| **セキュリティ仕様** | 具体的な実装方針が不足 | 高 |
| **パフォーマンス要件** | 詳細な指標・測定方法が未定義 | 中 |
| **運用・監視** | 実装レベルの詳細が不足 | 中 |
| **テスト仕様** | テストケース・カバレッジ基準が未定義 | 高 |
| **デプロイ戦略** | CI/CD、環境管理の詳細が不足 | 中 |

## 2. Claude Code開発指示のための追記事項

### 2.1 技術的実装詳細

#### 2.1.1 プロジェクト構成

```
rental-matching-app/
├── apps/
│   ├── web/                 # React MPA (Vite + TypeScript)
│   ├── api/                 # Node.js + Express API
│   └── line-bot/           # LINE Bot専用サービス
├── packages/
│   ├── shared/             # 共通型定義・ユーティリティ
│   ├── database/           # Prisma スキーマ・マイグレーション
│   ├── ui/                 # 共通UIコンポーネント（LINE Design Language）
│   └── i18n/              # 多言語リソース
├── libs/
│   ├── auth/              # 認証ライブラリ
│   ├── search/            # 検索エンジンライブラリ
│   ├── ai-chat/           # AI処理ライブラリ
│   └── external-api/      # 外部API連携ライブラリ
└── configs/
    ├── eslint/
    ├── typescript/
    └── tailwind/
```

#### 2.1.2 技術スタック詳細

| カテゴリ | 技術 | バージョン | 設定詳細 |
|----------|------|------------|----------|
| **フロントエンド** | React | 18.2+ | MPA構成、TypeScript strict mode |
| **ビルドツール** | Vite | 5.0+ | HMR、プリロード最適化、コード分割 |
| **スタイリング** | Tailwind CSS | 3.4+ | LINE Design Language準拠のカスタム設定 |
| **状態管理** | Zustand | 4.4+ | 軽量、TypeScript親和性 |
| **フォーム管理** | React Hook Form | 7.45+ | Zod バリデーション統合 |
| **API クライアント** | TanStack Query | 5.0+ | キャッシュ、リアルタイム更新 |
| **ルーティング** | React Router | 6.15+ | MPA対応、ファイルベースルーティング |

#### 2.1.3 データベース設定（Prisma）

```prisma
generator client {
  provider = "prisma-client-js"
  previewFeatures = ["postgresqlExtensions", "fullTextSearch"]
}

datasource db {
  provider = "postgresql"
  url = env("DATABASE_URL")
  extensions = [postgis, pg_trgm]
}
```

#### 2.1.4 環境変数設定

```typescript
// apps/api/src/config/env.ts
import { z } from 'zod';

const envSchema = z.object({
  NODE_ENV: z.enum(['development', 'production', 'test']),
  DATABASE_URL: z.string().url(),
  JWT_SECRET: z.string().min(32),
  JWT_REFRESH_SECRET: z.string().min(32),
  LINE_CHANNEL_SECRET: z.string(),
  LINE_CHANNEL_ACCESS_TOKEN: z.string(),
  GOOGLE_MAPS_API_KEY: z.string(),
  GOOGLE_TRANSLATE_API_KEY: z.string(),
  SMTP_HOST: z.string(),
  SMTP_PORT: z.coerce.number(),
  REDIS_URL: z.string().url(),
  // 外部サイト連携用
  JIMOTY_API_KEY: z.string().optional(),
  UCHIKOMI_API_KEY: z.string().optional(),
});

export const env = envSchema.parse(process.env);
```

### 2.2 セキュリティ実装詳細

#### 2.2.1 認証・認可

| 項目 | 実装方針 | 具体的設定 |
|------|----------|------------|
| **JWT設定** | RS256アルゴリズム | アクセストークン: 15分、リフレッシュトークン: 7日 |
| **パスワード** | bcrypt | saltRounds: 12, 最小8文字、英数字記号必須 |
| **レート制限** | express-rate-limit | API: 100req/15min, 認証: 5req/15min |
| **CORS設定** | 許可ドメイン限定 | 本番: 指定ドメインのみ、開発: localhost許可 |
| **CSP** | Content Security Policy | script-src, style-src, img-src制限 |

#### 2.2.2 データ保護

```typescript
// packages/shared/src/crypto.ts
import crypto from 'node:crypto';

export class DataEncryption {
  private static algorithm = 'aes-256-gcm';
  private static key = Buffer.from(process.env.ENCRYPTION_KEY, 'hex');
  
  static encrypt(text: string): EncryptedData {
    const iv = crypto.randomBytes(16);
    const cipher = crypto.createCipher(this.algorithm, this.key);
    // 実装詳細...
  }
}
```

### 2.3 パフォーマンス要件詳細

#### 2.3.1 指標・監視

| メトリクス | 目標値 | 測定方法 | アラート閾値 |
|------------|--------|----------|-------------|
| **ページ読込時間** | < 3秒 | Lighthouse CI | > 4秒で警告 |
| **API応答時間** | < 1秒 | APM監視 | > 2秒で警告 |
| **データベースクエリ** | < 500ms | Prisma metrics | > 1秒で警告 |
| **画像最適化** | WebP変換、遅延読込 | Bundle analyzer | 2MB超で警告 |
| **メモリ使用量** | < 512MB | Docker monitoring | > 1GB で警告 |

#### 2.3.2 キャッシュ戦略

```typescript
// libs/cache/src/redis-cache.ts
export class CacheManager {
  // 検索結果: 5分キャッシュ
  static readonly SEARCH_RESULTS_TTL = 300;
  // 物件詳細: 1時間キャッシュ  
  static readonly PROPERTY_DETAILS_TTL = 3600;
  // ユーザー設定: 24時間キャッシュ
  static readonly USER_SETTINGS_TTL = 86400;
}
```

### 2.4 LINE Design Language実装詳細

#### 2.4.1 Tailwind CSS設定

```javascript
// configs/tailwind/line-design.config.js
module.exports = {
  theme: {
    extend: {
      colors: {
        // LINE緑色系
        line: {
          50: '#f0f9f3',
          500: '#00c300', // メインカラー
          600: '#00a300',
          700: '#008000',
        },
        // 外国人向け配色
        accent: {
          blue: '#0084ff',    // 情報表示
          orange: '#ff8c00',  // 注意喚起
          red: '#ff3b30',     // エラー
        }
      },
      fontFamily: {
        sans: ['Hiragino Sans', 'ヒラギノ角ゴ ProN', 'Noto Sans JP', 'sans-serif'],
        en: ['Roboto', 'Arial', 'sans-serif'],
      },
      spacing: {
        'line-xs': '4px',
        'line-sm': '8px', 
        'line-md': '16px',
        'line-lg': '24px',
        'line-xl': '32px',
      }
    }
  },
  plugins: [
    require('@tailwindcss/forms'),
    require('@tailwindcss/typography'),
  ]
}
```

#### 2.4.2 コンポーネント設計原則

```typescript
// packages/ui/src/components/Button.tsx
interface ButtonProps {
  variant: 'primary' | 'secondary' | 'ghost' | 'danger';
  size: 'sm' | 'md' | 'lg';
  fullWidth?: boolean;
  loading?: boolean;
  // LINE Design Language準拠のリップル効果
  ripple?: boolean;
}
```

### 2.5 多言語対応実装詳細

#### 2.5.1 i18n設定

```typescript
// packages/i18n/src/config.ts
export const supportedLanguages = [
  'ja', 'en', 'zh-CN', 'zh-TW', 'ko', 'vi', 'ne', 'tl'
] as const;

export const i18nConfig = {
  defaultLanguage: 'ja',
  fallbackLanguage: 'en',
  // 自動翻訳のしきい値
  autoTranslateThreshold: 0.8,
  // RTL言語対応（将来のアラビア語等）
  rtlLanguages: [],
};
```

#### 2.5.2 翻訳リソース構造

```
packages/i18n/locales/
├── ja/
│   ├── common.json          # 共通UI要素
│   ├── auth.json           # 認証関連
│   ├── property.json       # 物件関連
│   ├── chat.json          # AI Chat関連
│   └── validation.json    # バリデーションメッセージ
├── en/
├── zh-CN/
└── ...
```

### 2.6 テスト戦略詳細

#### 2.6.1 テスト分類・カバレッジ目標

| テスト種別 | フレームワーク | カバレッジ目標 | 実行タイミング |
|------------|----------------|----------------|----------------|
| **単体テスト** | Vitest | 90%以上 | 各PR |
| **結合テスト** | Testing Library | 80%以上 | 各PR |
| **E2Eテスト** | Playwright | 主要フロー100% | リリース前 |
| **API テスト** | Supertest | 95%以上 | 各PR |
| **パフォーマンステスト** | k6 | 基準値クリア | 週次 |

#### 2.6.2 テストデータ管理

```typescript
// packages/database/src/test-data/seed.ts
export const testUsers = {
  landlord: {
    email: 'landlord@test.com',
    userType: 'landlord' as const,
    // 暗号化された固定パスワード
  },
  foreignTenant: {
    email: 'tenant@test.com', 
    userType: 'tenant' as const,
    nationality: 'vietnam',
    preferredLanguage: 'vi',
  }
};
```

### 2.7 CI/CD・デプロイ戦略

#### 2.7.1 GitHub Actions ワークフロー

```yaml
# .github/workflows/ci.yml
name: CI/CD Pipeline
on:
  push:
    branches: [main, develop]
  pull_request:
    branches: [main]

jobs:
  test:
    runs-on: ubuntu-latest
    services:
      postgres:
        image: postgis/postgis:15-3.3
        env:
          POSTGRES_PASSWORD: test
    steps:
      - uses: actions/checkout@v4
      - uses: actions/setup-node@v4
        with:
          node-version: '18'
          cache: 'pnpm'
      
      - name: Install dependencies
        run: pnpm install --frozen-lockfile
      
      - name: Run type checking
        run: pnpm type-check
      
      - name: Run linting
        run: pnpm lint
      
      - name: Run unit tests
        run: pnpm test:unit --coverage
      
      - name: Run E2E tests
        run: pnpm test:e2e
```

#### 2.7.2 環境構成

| 環境 | 用途 | インフラ | デプロイ方法 |
|------|------|----------|------------|
| **development** | 開発・テスト | Docker Compose | 手動 |
| **staging** | QA・受入テスト | AWS ECS | 自動（develop ブランチ） |
| **production** | 本番運用 | AWS ECS | 手動承認後（main ブランチ） |

### 2.8 監視・ログ戦略

#### 2.8.1 ログ設定

```typescript
// libs/logger/src/winston.config.ts
import winston from 'winston';

export const logger = winston.createLogger({
  level: process.env.LOG_LEVEL || 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.errors({ stack: true }),
    winston.format.json()
  ),
  defaultMeta: { 
    service: process.env.SERVICE_NAME,
    version: process.env.APP_VERSION 
  },
  transports: [
    new winston.transports.Console(),
    new winston.transports.File({ 
      filename: 'logs/error.log', 
      level: 'error' 
    }),
    new winston.transports.File({ 
      filename: 'logs/combined.log' 
    })
  ]
});
```

#### 2.8.2 メトリクス・アラート

```typescript
// libs/monitoring/src/metrics.ts
export const metrics = {
  // ビジネスメトリクス
  propertyViews: new Counter('property_views_total'),
  inquirySent: new Counter('inquiry_sent_total'),
  userRegistrations: new Counter('user_registrations_total'),
  
  // 技術メトリクス  
  apiDuration: new Histogram('api_duration_seconds'),
  dbQueryDuration: new Histogram('db_query_duration_seconds'),
  cacheHitRate: new Gauge('cache_hit_rate'),
};
```

### 2.9 データマイグレーション戦略

#### 2.9.1 Prisma マイグレーション

```typescript
// packages/database/prisma/migrations/
// ゼロダウンタイム原則
// 1. カラム追加 → 2. データ移行 → 3. 旧カラム削除
// の3段階でマイグレーション実行
```

#### 2.9.2 シードデータ

```typescript
// packages/database/src/seed/production.ts
// 本番環境用の基本データ
export const productionSeed = {
  amenities: [
    { code: 'wifi', nameJa: 'Wi-Fi', nameEn: 'Wi-Fi' },
    { code: 'aircon', nameJa: 'エアコン', nameEn: 'Air Conditioning' },
    // ... 50種類の設備データ
  ],
  faqs: [
    // 8言語 × 20カテゴリ × 10質問 = 1600件のFAQデータ
  ]
};
```

## 3. Claude Code指示時の重要ポイント

### 3.1 開発優先順位

```
Phase 1 (MVP): 4ヶ月
├── 認証システム（多要素認証含む）
├── 物件基本管理（手動登録）
├── 検索・詳細表示
├── 多言語対応（8言語）
└── 基本的な問い合わせ機能

Phase 2 (差別化機能): 3ヶ月  
├── AI Chat（LINE Bot統合）
├── 外部サイト連携
├── PDF自動取込
├── 地図機能強化
└── 分析ダッシュボード

Phase 3 (最適化): 2ヶ月
├── パフォーマンス最適化
├── 高度な検索機能
├── 通知システム強化
└── 運用ツール整備
```

### 3.2 技術的制約・注意事項

| 項目 | 制約・注意点 |
|------|------------|
| **外部API制限** | Google Maps: 月間40,000リクエスト制限 |
| **スクレイピング** | robots.txt遵守、レート制限（1req/sec） |
| **ファイルアップロード** | 画像: 10MB、PDF: 50MB上限 |
| **データベース** | PostgreSQL 15+、PostGIS拡張必須 |
| **メモリ使用量** | NODE_OPTIONS="--max-old-space-size=4096" |

### 3.3 品質基準

```typescript
// package.json scripts
{
  "type-check": "tsc --noEmit",
  "lint": "eslint . --ext .ts,.tsx",
  "lint:fix": "eslint . --ext .ts,.tsx --fix", 
  "format": "prettier --write .",
  "test:unit": "vitest run",
  "test:e2e": "playwright test",
  "test:coverage": "vitest run --coverage",
  "build": "vite build",
  "preview": "vite preview"
}
```

## 4. 推奨実装順序

### 4.1 Week 1-2: プロジェクトセットアップ
1. モノレポ構成（pnpm workspace）
2. TypeScript設定（strict mode）
3. ESLint + Prettier設定
4. Tailwind + LINE Design Language設定
5. データベース設計・マイグレーション

### 4.2 Week 3-6: 認証・基本機能
1. 認証システム（JWT + 多要素認証）
2. ユーザー管理（プロフィール、言語設定）
3. 物件手動登録機能
4. 基本検索・一覧表示

### 4.3 Week 7-10: 検索・表示機能
1. 高度検索機能
2. 地図統合
3. 物件詳細ページ
4. お気に入り機能

### 4.4 Week 11-14: コミュニケーション機能
1. 問い合わせシステム
2. 通知機能
3. 入居申込みフロー
4. 大家ダッシュボード

### 4.5 Week 15-16: 多言語対応・テスト
1. 8言語リソース整備
2. 自動翻訳統合
3. E2Eテスト整備
4. パフォーマンステスト

この詳細仕様により、Claude Codeに対して具体的で実装可能な指示を出すことができます。