# 技術アーキテクチャ設計書

## 1. 技術スタック概要

| 分類 | 技術 | 用途 | 備考 |
|------|------|------|------|
| フロントエンド | React 18+ | UI構築 | MPA（Multi Page Application） |
| 言語 | TypeScript | 型安全な開発 | strict mode |
| スタイリング | Tailwind CSS | デザインシステム | LINE Design Language準拠 |
| ビルドツール | Vite | 開発環境・ビルド | 高速開発 |
| データベース | PostgreSQL | データストレージ | 多言語対応 |
| ORM | Prisma | データベースアクセス | TypeScript統合 |
| バックエンド | Node.js + Express | API開発 | RESTful API |

## 2. アーキテクチャ構成

### 2.1 コア機能モジュール

| モジュールID | モジュール名 | 概要 | 関連機能 |
|-------------|------------|------|----------|
| CORE-001 | 認証モジュール | ユーザー認証・認可 | FUNC-001〜006 |
| CORE-002 | 物件管理モジュール | 物件CRUD操作 | FUNC-007〜013 |
| CORE-003 | 検索モジュール | 物件検索エンジン | FUNC-014〜019 |
| CORE-004 | 多言語モジュール | 国際化対応 | FUNC-031〜033 |
| CORE-005 | 通知モジュール | メール・プッシュ通知 | FUNC-024 |

### 2.2 拡張機能モジュール

| モジュールID | モジュール名 | 概要 | 関連機能 |
|-------------|------------|------|----------|
| EXT-001 | AI Chat モジュール | LINE Bot・AI相談 | FUNC-025〜030 |
| EXT-002 | 外部連携モジュール | 外部サイト連携 | FUNC-008〜009 |
| EXT-003 | 分析モジュール | データ分析・レポート | FUNC-034〜036 |
| EXT-004 | 地図連携モジュール | Google Maps統合 | FUNC-015, FUNC-018 |
| EXT-005 | ファイル処理モジュール | 画像・PDF処理 | FUNC-009〜010 |

## 3. 関数一覧

### 3.1 認証モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| AUTH-001 | registerUser | ユーザー登録 | UserRegistrationData | Promise<User> | PROC-001 |
| AUTH-002 | authenticateUser | ユーザー認証 | email: string, password: string | Promise<AuthResult> | PROC-002 |
| AUTH-003 | generateToken | JWTトークン生成 | userId: string, role: UserRole | string | PROC-002 |
| AUTH-004 | verifyToken | トークン検証 | token: string | Promise<TokenPayload> | PROC-005 |
| AUTH-005 | hashPassword | パスワードハッシュ化 | password: string | Promise<string> | PROC-004 |
| AUTH-006 | comparePassword | パスワード照合 | password: string, hash: string | Promise<boolean> | PROC-002 |
| AUTH-007 | sendVerificationCode | 認証コード送信 | phone: string, email: string | Promise<void> | PROC-006 |
| AUTH-008 | verifyCode | 認証コード検証 | code: string, identifier: string | Promise<boolean> | PROC-006 |
| AUTH-009 | checkPermission | 権限チェック | userId: string, resource: string | Promise<boolean> | PROC-007 |

### 3.2 物件管理モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| PROP-001 | createProperty | 物件作成 | PropertyData | Promise<Property> | PROC-008 |
| PROP-002 | updateProperty | 物件更新 | id: string, data: Partial<PropertyData> | Promise<Property> | PROC-009 |
| PROP-003 | deleteProperty | 物件削除 | id: string | Promise<void> | PROC-010 |
| PROP-004 | getProperty | 物件取得 | id: string | Promise<Property> | PROC-015 |
| PROP-005 | getPropertiesByOwner | オーナー物件一覧 | ownerId: string | Promise<Property[]> | PROC-015 |
| PROP-006 | uploadPropertyImages | 物件画像アップロード | propertyId: string, files: File[] | Promise<PropertyImage[]> | PROC-011 |
| PROP-007 | optimizeImage | 画像最適化 | imageBuffer: Buffer | Promise<Buffer> | PROC-012 |
| PROP-008 | togglePropertyStatus | 公開状態切替 | id: string, status: PropertyStatus | Promise<Property> | PROC-009 |
| PROP-009 | validatePropertyData | 物件データ検証 | data: PropertyData | ValidationResult | PROC-008 |

### 3.3 検索モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| SEARCH-001 | searchProperties | 物件検索 | SearchCriteria | Promise<SearchResult> | PROC-015 |
| SEARCH-002 | buildSearchQuery | 検索クエリ構築 | criteria: SearchCriteria | QueryBuilder | PROC-015 |
| SEARCH-003 | sortResults | 検索結果ソート | results: Property[], sortBy: SortOption | Property[] | PROC-016 |
| SEARCH-004 | paginateResults | ページネーション | results: Property[], page: number, limit: number | PaginatedResult | PROC-016 |
| SEARCH-005 | cacheSearchResult | 検索結果キャッシュ | key: string, data: SearchResult | Promise<void> | PROC-019 |
| SEARCH-006 | getCachedResult | キャッシュ取得 | key: string | Promise<SearchResult \| null> | PROC-019 |
| SEARCH-007 | searchNearbyFacilities | 周辺施設検索 | lat: number, lng: number, radius: number | Promise<Facility[]> | PROC-018 |
| SEARCH-008 | generateSearchSuggestions | 検索候補生成 | query: string | Promise<string[]> | PROC-015 |

### 3.4 多言語モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| I18N-001 | detectLanguage | 言語判定 | request: Request | string | PROC-041 |
| I18N-002 | loadTranslations | 翻訳リソース読込 | language: string | Promise<TranslationResource> | PROC-042 |
| I18N-003 | translateText | テキスト翻訳 | text: string, from: string, to: string | Promise<string> | PROC-023 |
| I18N-004 | formatCurrency | 通貨フォーマット | amount: number, currency: string, locale: string | string | PROC-044 |
| I18N-005 | formatDate | 日付フォーマット | date: Date, locale: string | string | PROC-044 |
| I18N-006 | validateLocale | ロケール検証 | locale: string | boolean | PROC-041 |
| I18N-007 | getLocalizedContent | ローカライズコンテンツ取得 | contentId: string, locale: string | Promise<LocalizedContent> | PROC-042 |

### 3.5 AI Chat モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| AI-001 | processUserMessage | ユーザーメッセージ処理 | message: string, userId: string, language: string | Promise<AIResponse> | PROC-026 |
| AI-002 | analyzeIntent | 意図解析 | message: string, language: string | Promise<Intent> | PROC-026 |
| AI-003 | generateResponse | 応答生成 | intent: Intent, context: ChatContext | Promise<string> | PROC-027 |
| AI-004 | searchFAQ | FAQ検索 | query: string, language: string | Promise<FAQResult[]> | PROC-030 |
| AI-005 | getPropertyAdvice | 物件アドバイス | propertyId: string, userQuery: string | Promise<string> | PROC-027 |
| AI-006 | getAreaInformation | 地域情報取得 | location: Location, language: string | Promise<AreaInfo> | PROC-027 |
| AI-007 | saveChatHistory | チャット履歴保存 | userId: string, message: string, response: string | Promise<void> | PROC-029 |
| AI-008 | handleLINEWebhook | LINE Webhook処理 | webhookData: LINEWebhookEvent | Promise<void> | PROC-024 |

### 3.6 外部連携モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| EXT-001 | scrapeJimoty | ジモティスクレイピング | url: string | Promise<PropertyData> | PROC-020 |
| EXT-002 | scrapeUchikomi | ウチコミスクレイピング | url: string | Promise<PropertyData> | PROC-020 |
| EXT-003 | scrapeEchoes | エコーズスクレイピング | url: string | Promise<PropertyData> | PROC-020 |
| EXT-004 | processPDF | PDF解析処理 | pdfBuffer: Buffer | Promise<PropertyData> | PROC-021 |
| EXT-005 | extractTextFromPDF | PDF文字抽出 | pdfBuffer: Buffer | Promise<string> | PROC-021 |
| EXT-006 | parsePropertyInfo | 物件情報パース | text: string | PropertyData | PROC-021 |
| EXT-007 | syncExternalData | 外部データ同期 | source: ExternalSource | Promise<SyncResult> | PROC-025 |
| EXT-008 | validateExternalData | 外部データ検証 | data: any, source: ExternalSource | ValidationResult | PROC-020 |

### 3.7 地図連携モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| MAP-001 | geocodeAddress | 住所ジオコーディング | address: string | Promise<Coordinates> | PROC-017 |
| MAP-002 | reverseGeocode | 逆ジオコーディング | lat: number, lng: number | Promise<Address> | PROC-017 |
| MAP-003 | calculateDistance | 距離計算 | from: Coordinates, to: Coordinates | number | PROC-018 |
| MAP-004 | findNearbyStations | 最寄り駅検索 | coordinates: Coordinates | Promise<Station[]> | PROC-018 |
| MAP-005 | getMapTileUrl | 地図タイルURL生成 | zoom: number, x: number, y: number | string | PROC-017 |
| MAP-006 | searchPlaces | 施設検索 | query: string, location: Coordinates | Promise<Place[]> | PROC-018 |

### 3.8 通知モジュール関数

| 関数ID | 関数名 | 概要 | 引数 | 戻り値 | 関連処理 |
|--------|--------|------|------|--------|----------|
| NOTIFY-001 | sendEmail | メール送信 | to: string, subject: string, body: string | Promise<void> | PROC-031 |
| NOTIFY-002 | sendPushNotification | プッシュ通知送信 | userId: string, message: string | Promise<void> | PROC-032 |
| NOTIFY-003 | sendSMS | SMS送信 | phone: string, message: string | Promise<void> | PROC-006 |
| NOTIFY-004 | createNotification | 通知作成 | NotificationData | Promise<Notification> | PROC-031 |
| NOTIFY-005 | markAsRead | 既読処理 | notificationId: string | Promise<void> | PROC-034 |
| NOTIFY-006 | getUnreadCount | 未読数取得 | userId: string | Promise<number> | PROC-034 |

## 4. データベース設計

### 4.1 テーブル定義

#### 4.1.1 ユーザー関連テーブル

```sql
-- ユーザーテーブル
CREATE TABLE users (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    user_type VARCHAR(20) NOT NULL CHECK (user_type IN ('landlord', 'tenant')),
    phone VARCHAR(20),
    preferred_language VARCHAR(5) DEFAULT 'ja',
    email_verified BOOLEAN DEFAULT FALSE,
    phone_verified BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- ユーザープロフィールテーブル
CREATE TABLE user_profiles (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    first_name VARCHAR(100),
    last_name VARCHAR(100),
    first_name_kana VARCHAR(100),
    last_name_kana VARCHAR(100),
    birth_date DATE,
    gender VARCHAR(10),
    nationality VARCHAR(50),
    visa_status VARCHAR(50),
    occupation VARCHAR(100),
    annual_income INTEGER,
    profile_image_url TEXT,
    self_introduction TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 認証コードテーブル
CREATE TABLE verification_codes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    code VARCHAR(10) NOT NULL,
    type VARCHAR(20) NOT NULL CHECK (type IN ('email', 'phone')),
    expires_at TIMESTAMP NOT NULL,
    used_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

#### 4.1.2 物件関連テーブル

```sql
-- 物件テーブル
CREATE TABLE properties (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    owner_id UUID REFERENCES users(id) ON DELETE CASCADE,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    property_type VARCHAR(50) NOT NULL,
    structure_type VARCHAR(50),
    floors_total INTEGER,
    floor_location INTEGER,
    built_year INTEGER,
    rent_amount INTEGER NOT NULL,
    management_fee INTEGER DEFAULT 0,
    deposit INTEGER DEFAULT 0,
    key_money INTEGER DEFAULT 0,
    room_layout VARCHAR(20),
    area_sqm DECIMAL(6,2),
    address_prefecture VARCHAR(20) NOT NULL,
    address_city VARCHAR(50) NOT NULL,
    address_town VARCHAR(100),
    address_detail VARCHAR(255),
    postal_code VARCHAR(10),
    latitude DECIMAL(10,8),
    longitude DECIMAL(11,8),
    nearest_station VARCHAR(100),
    walking_minutes INTEGER,
    availability_status VARCHAR(20) DEFAULT 'available' 
        CHECK (availability_status IN ('available', 'pending', 'occupied', 'unavailable')),
    is_published BOOLEAN DEFAULT FALSE,
    external_source VARCHAR(50),
    external_id VARCHAR(100),
    external_url TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 物件画像テーブル
CREATE TABLE property_images (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    property_id UUID REFERENCES properties(id) ON DELETE CASCADE,
    image_url TEXT NOT NULL,
    image_type VARCHAR(20) DEFAULT 'interior',
    display_order INTEGER DEFAULT 1,
    caption TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 物件設備テーブル
CREATE TABLE property_amenities (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    property_id UUID REFERENCES properties(id) ON DELETE CASCADE,
    amenity_code VARCHAR(50) NOT NULL,
    amenity_name VARCHAR(100) NOT NULL,
    category VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 物件多言語情報テーブル
CREATE TABLE property_translations (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    property_id UUID REFERENCES properties(id) ON DELETE CASCADE,
    language_code VARCHAR(5) NOT NULL,
    title VARCHAR(255),
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(property_id, language_code)
);
```

#### 4.1.3 コミュニケーション関連テーブル

```sql
-- 問い合わせテーブル
CREATE TABLE inquiries (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    property_id UUID REFERENCES properties(id) ON DELETE CASCADE,
    tenant_id UUID REFERENCES users(id) ON DELETE CASCADE,
    landlord_id UUID REFERENCES users(id) ON DELETE CASCADE,
    subject VARCHAR(255),
    message TEXT NOT NULL,
    status VARCHAR(20) DEFAULT 'open' 
        CHECK (status IN ('open', 'replied', 'closed')),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 問い合わせ返信テーブル
CREATE TABLE inquiry_replies (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    inquiry_id UUID REFERENCES inquiries(id) ON DELETE CASCADE,
    sender_id UUID REFERENCES users(id) ON DELETE CASCADE,
    message TEXT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 入居申込みテーブル
CREATE TABLE applications (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    property_id UUID REFERENCES properties(id) ON DELETE CASCADE,
    tenant_id UUID REFERENCES users(id) ON DELETE CASCADE,
    landlord_id UUID REFERENCES users(id) ON DELETE CASCADE,
    status VARCHAR(20) DEFAULT 'pending' 
        CHECK (status IN ('pending', 'approved', 'rejected', 'contracted')),
    desired_move_in_date DATE,
    guarantor_name VARCHAR(100),
    guarantor_phone VARCHAR(20),
    emergency_contact_name VARCHAR(100),
    emergency_contact_phone VARCHAR(20),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 通知テーブル
CREATE TABLE notifications (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    title VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    type VARCHAR(50) NOT NULL,
    is_read BOOLEAN DEFAULT FALSE,
    action_url TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

#### 4.1.4 AI・チャット関連テーブル

```sql
-- チャット履歴テーブル
CREATE TABLE chat_histories (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    session_id VARCHAR(100),
    platform VARCHAR(20) DEFAULT 'web' CHECK (platform IN ('web', 'line')),
    user_message TEXT NOT NULL,
    ai_response TEXT NOT NULL,
    intent VARCHAR(100),
    language_code VARCHAR(5) DEFAULT 'ja',
    property_id UUID REFERENCES properties(id) ON DELETE SET NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- FAQテーブル
CREATE TABLE faqs (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    category VARCHAR(50) NOT NULL,
    question TEXT NOT NULL,
    answer TEXT NOT NULL,
    language_code VARCHAR(5) DEFAULT 'ja',
    is_active BOOLEAN DEFAULT TRUE,
    display_order INTEGER DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- LINE連携テーブル
CREATE TABLE line_users (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    line_user_id VARCHAR(100) UNIQUE NOT NULL,
    display_name VARCHAR(100),
    language VARCHAR(10) DEFAULT 'ja',
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

#### 4.1.5 システム管理関連テーブル

```sql
-- お気に入りテーブル
CREATE TABLE favorites (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    property_id UUID REFERENCES properties(id) ON DELETE CASCADE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, property_id)
);

-- 検索履歴テーブル
CREATE TABLE search_histories (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    search_query JSONB NOT NULL,
    result_count INTEGER,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- システムログテーブル
CREATE TABLE system_logs (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    level VARCHAR(20) NOT NULL,
    message TEXT NOT NULL,
    user_id UUID REFERENCES users(id) ON DELETE SET NULL,
    ip_address INET,
    user_agent TEXT,
    request_url TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 設定テーブル
CREATE TABLE system_settings (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    key VARCHAR(100) UNIQUE NOT NULL,
    value TEXT,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### 4.2 インデックス定義

```sql
-- パフォーマンス向上のためのインデックス
CREATE INDEX idx_properties_location ON properties(latitude, longitude);
CREATE INDEX idx_properties_rent ON properties(rent_amount);
CREATE INDEX idx_properties_status ON properties(availability_status, is_published);
CREATE INDEX idx_properties_area ON properties(address_prefecture, address_city);
CREATE INDEX idx_properties_owner ON properties(owner_id);
CREATE INDEX idx_inquiries_status ON inquiries(status, created_at);
CREATE INDEX idx_notifications_user_unread ON notifications(user_id, is_read);
CREATE INDEX idx_chat_histories_user_session ON chat_histories(user_id, session_id);
CREATE INDEX idx_favorites_user ON favorites(user_id);

-- 全文検索用インデックス
CREATE INDEX idx_properties_fulltext ON properties USING gin(to_tsvector('japanese', title || ' ' || COALESCE(description, '')));
```

## 5. ER図

```mermaid
erDiagram
    users ||--o{ user_profiles : has
    users ||--o{ properties : owns
    users ||--o{ inquiries : sends
    users ||--o{ inquiries : receives
    users ||--o{ applications : submits
    users ||--o{ notifications : receives
    users ||--o{ chat_histories : has
    users ||--o{ favorites : bookmarks
    users ||--o{ search_histories : searches
    users ||--o{ line_users : links
    users ||--o{ verification_codes : receives

    properties ||--o{ property_images : contains
    properties ||--o{ property_amenities : has
    properties ||--o{ property_translations : translated_to
    properties ||--o{ inquiries : about
    properties ||--o{ applications : for
    properties ||--o{ favorites : bookmarked_as
    properties ||--o{ chat_histories : discussed

    inquiries ||--o{ inquiry_replies : has

    users {
        uuid id PK
        string email
        string password_hash
        string user_type
        string phone
        string preferred_language
        boolean email_verified
        boolean phone_verified
        boolean is_active
        timestamp created_at
        timestamp updated_at
    }

    user_profiles {
        uuid id PK
        uuid user_id FK
        string first_name
        string last_name
        string first_name_kana
        string last_name_kana
        date birth_date
        string gender
        string nationality
        string visa_status
        string occupation
        integer annual_income
        text profile_image_url
        text self_introduction
        timestamp created_at
        timestamp updated_at
    }

    properties {
        uuid id PK
        uuid owner_id FK
        string title
        text description
        string property_type
        string structure_type
        integer floors_total
        integer floor_location
        integer built_year
        integer rent_amount
        integer management_fee
        integer deposit
        integer key_money
        string room_layout
        decimal area_sqm
        string address_prefecture
        string address_city
        string address_town
        string address_detail
        string postal_code
        decimal latitude
        decimal longitude
        string nearest_station
        integer walking_minutes
        string availability_status
        boolean is_published
        string external_source
        string external_id
        text external_url
        timestamp created_at
        timestamp updated_at
    }

    property_images {
        uuid id PK
        uuid property_id FK
        text image_url
        string image_type
        integer display_order
        text caption
        timestamp created_at
    }

    property_amenities {
        uuid id PK
        uuid property_id FK
        string amenity_code
        string amenity_name
        string category
        timestamp created_at
    }

    property_translations {
        uuid id PK
        uuid property_id FK
        string language_code
        string title
        text description
        timestamp created_at
    }

    inquiries {
        uuid id PK
        uuid property_id FK
        uuid tenant_id FK
        uuid landlord_id FK
        string subject
        text message
        string status
        timestamp created_at
        timestamp updated_at
    }

    inquiry_replies {
        uuid id PK
        uuid inquiry_id FK
        uuid sender_id FK
        text message
        timestamp created_at
    }

    applications {
        uuid id PK
        uuid property_id FK
        uuid tenant_id FK
        uuid landlord_id FK
        string status
        date desired_move_in_date
        string guarantor_name
        string guarantor_phone
        string emergency_contact_name
        string emergency_contact_phone
        text notes
        timestamp created_at
        timestamp updated_at
    }

    notifications {
        uuid id PK
        uuid user_id FK
        string title
        text message
        string type
        boolean is_read
        text action_url
        timestamp created_at
    }

    chat_histories {
        uuid id PK
        uuid user_id FK
        string session_id
        string platform
        text user_message
        text ai_response
        string intent
        string language_code
        uuid property_id FK
        timestamp created_at
    }

    faqs {
        uuid id PK
        string category
        text question
        text answer
        string language_code
        boolean is_active
        integer display_order
        timestamp created_at
        timestamp updated_at
    }

    line_users {
        uuid id PK
        uuid user_id FK
        string line_user_id
        string display_name
        string language
        boolean is_active
        timestamp created_at
    }

    favorites {
        uuid id PK
        uuid user_id FK
        uuid property_id FK
        timestamp created_at
    }

    search_histories {
        uuid id PK
        uuid user_id FK
        jsonb search_query
        integer result_count
        timestamp created_at
    }

    verification_codes {
        uuid id PK
        uuid user_id FK
        string code
        string type
        timestamp expires_at
        timestamp used_at
        timestamp created_at
    }

    system_logs {
        uuid id PK
        string level
        text message
        uuid user_id FK
        inet ip_address
        text user_agent
        text request_url
        timestamp created_at
    }

    system_settings {
        uuid id PK
        string key
        text value
        text description
        timestamp created_at
        timestamp updated_at
    }
```

## 6. APIエンドポイント設計

### 6.1 認証API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-001 | /api/auth/register | POST | ユーザー登録 | AUTH-001 |
| API-002 | /api/auth/login | POST | ログイン | AUTH-002 |
| API-003 | /api/auth/logout | POST | ログアウト | AUTH-003 |
| API-004 | /api/auth/verify-email | POST | メール認証 | AUTH-008 |
| API-005 | /api/auth/verify-phone | POST | 電話認証 | AUTH-008 |
| API-006 | /api/auth/forgot-password | POST | パスワードリセット | AUTH-007 |
| API-007 | /api/auth/reset-password | POST | パスワード再設定 | AUTH-004 |
| API-008 | /api/auth/refresh-token | POST | トークン更新 | AUTH-003 |
| API-009 | /api/auth/profile | GET | プロフィール取得 | AUTH-009 |
| API-010 | /api/auth/profile | PUT | プロフィール更新 | AUTH-009 |

### 6.2 物件API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-011 | /api/properties | GET | 物件検索 | SEARCH-001 |
| API-012 | /api/properties | POST | 物件作成 | PROP-001 |
| API-013 | /api/properties/:id | GET | 物件詳細 | PROP-004 |
| API-014 | /api/properties/:id | PUT | 物件更新 | PROP-002 |
| API-015 | /api/properties/:id | DELETE | 物件削除 | PROP-003 |
| API-016 | /api/properties/:id/images | POST | 画像アップロード | PROP-006 |
| API-017 | /api/properties/:id/images/:imageId | DELETE | 画像削除 | PROP-007 |
| API-018 | /api/properties/:id/status | PUT | 公開状態変更 | PROP-008 |
| API-019 | /api/properties/external/import | POST | 外部サイト取込 | EXT-001, EXT-002, EXT-003 |
| API-020 | /api/properties/pdf/import | POST | PDF取込 | EXT-004 |
| API-021 | /api/properties/nearby | GET | 周辺物件検索 | SEARCH-007 |
| API-022 | /api/properties/:id/facilities | GET | 周辺施設検索 | MAP-006 |

### 6.3 お気に入り・検索API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-023 | /api/favorites | GET | お気に入り一覧 | SEARCH-001 |
| API-024 | /api/favorites | POST | お気に入り追加 | PROP-004 |
| API-025 | /api/favorites/:id | DELETE | お気に入り削除 | PROP-004 |
| API-026 | /api/search/suggestions | GET | 検索候補 | SEARCH-008 |
| API-027 | /api/search/history | GET | 検索履歴 | SEARCH-001 |

### 6.4 問い合わせ・申込みAPI

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-028 | /api/inquiries | POST | 問い合わせ送信 | NOTIFY-004 |
| API-029 | /api/inquiries | GET | 問い合わせ一覧 | NOTIFY-004 |
| API-030 | /api/inquiries/:id | GET | 問い合わせ詳細 | NOTIFY-004 |
| API-031 | /api/inquiries/:id/replies | POST | 返信送信 | NOTIFY-004 |
| API-032 | /api/applications | POST | 入居申込み | NOTIFY-004 |
| API-033 | /api/applications | GET | 申込み一覧 | NOTIFY-004 |
| API-034 | /api/applications/:id | GET | 申込み詳細 | NOTIFY-004 |
| API-035 | /api/applications/:id/status | PUT | 申込み状態更新 | NOTIFY-004 |

### 6.5 AI Chat API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-036 | /api/chat/message | POST | メッセージ送信 | AI-001 |
| API-037 | /api/chat/history | GET | チャット履歴 | AI-007 |
| API-038 | /api/chat/session | POST | セッション開始 | AI-007 |
| API-039 | /api/line/webhook | POST | LINE Webhook | AI-008 |
| API-040 | /api/faq | GET | FAQ一覧 | AI-004 |
| API-041 | /api/faq/search | GET | FAQ検索 | AI-004 |

### 6.6 通知API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-042 | /api/notifications | GET | 通知一覧 | NOTIFY-006 |
| API-043 | /api/notifications/:id/read | PUT | 既読処理 | NOTIFY-005 |
| API-044 | /api/notifications/unread-count | GET | 未読数取得 | NOTIFY-006 |

### 6.7 地図・位置情報API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-045 | /api/geocode | GET | 住所ジオコーディング | MAP-001 |
| API-046 | /api/reverse-geocode | GET | 逆ジオコーディング | MAP-002 |
| API-047 | /api/stations/nearby | GET | 最寄り駅検索 | MAP-004 |
| API-048 | /api/places/search | GET | 施設検索 | MAP-006 |

### 6.8 多言語・設定API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-049 | /api/translations/:language | GET | 翻訳リソース取得 | I18N-002 |
| API-050 | /api/translate | POST | テキスト翻訳 | I18N-003 |
| API-051 | /api/settings/language | PUT | 言語設定更新 | I18N-006 |

### 6.9 管理・分析API

| API ID | エンドポイント | メソッド | 概要 | 関連関数 |
|--------|---------------|----------|------|----------|
| API-052 | /api/dashboard/landlord | GET | 大家ダッシュボード | PROP-005 |
| API-053 | /api/analytics/properties | GET | 物件分析 | PROP-005 |
| API-054 | /api/analytics/inquiries | GET | 問い合わせ分析 | NOTIFY-004 |

この技術アーキテクチャ設計により、モジュール化された柔軟で拡張性の高いシステムを構築できます。